/* Thurlby-Thandar PLP series power supplies instrument driver */
/* Version 3.0 */
/* 27 Feb 1992 */
/* last updated June, 1998 */

#include        "gpib.h"
#include        "formatio.h"
#include        "tplp.h"

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* tplp_err: the error variable for the instrument module                  */
/*=========================================================================*/
static int address[tplp_MAX_INSTR + 1];
static int bd[tplp_MAX_INSTR + 1];
static int instr_cnt;
static int tplp_err;

/*= INTERNAL DATA =========================================================*/
/* cmd is a buffer for GPIB I/O strings */
static char buffer[100];

/*= UTILITY ROUTINES ======================================================*/
int tplp_open_instr (int);
int tplp_close_instr (int);
int tplp_invalid_integer_range (int, int, int, int);
int tplp_device_closed (int);
int tplp_read_data (int, char *, int);
int tplp_write_data (int, char *, int);

/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries the instrument    */
/*           for its ID, and initializes the instrument to a known state.  */
/*=========================================================================*/
int __stdcall tplp_init (addr, rest, instrID)
int addr;
int rest;
int *instrID;
{
    int  ID;

    if (tplp_invalid_integer_range (addr, 0, 30, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (rest, 0, 1, -3) != 0)
        return tplp_err;

    ID = tplp_open_instr (addr);
    if (ID <= 0)
        return tplp_err;

    if (tplp_write_data (ID, "*IDN?\012", 6) != 0) {
        tplp_close_instr (ID);
        return tplp_err;
        }
    if (tplp_read_data (ID, buffer, 50) != 0) {
        tplp_close_instr (ID);
        return tplp_err;
        }
    Scan (buffer, "THURLBY-THANDAR,PL");
    if (NumFmtdBytes () != 18)
        {
        tplp_err = 223;
        tplp_close_instr (ID);
        return tplp_err;
        }

    if (rest) {
        if (tplp_write_data (ID, "*RST\012", 5) != 0) {
            tplp_close_instr (ID);
            return tplp_err;
        }
    }
    *instrID = ID;
    return tplp_err;
}
/*=========================================================================*/
/* Function: Set Volts                                                     */
/* Purpose:  Set the Voltage of the selected output                        */
/*=========================================================================*/
int __stdcall tplp_set_volts (instrID,opno,voltage)
int instrID;
int opno;
double voltage;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (opno, 1, 3, -2) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"V%d %f\012",opno,voltage);
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return(tplp_err);
    return(tplp_err);  /* success */
}
/*=========================================================================*/
/* Function: Set current                                                   */
/* Purpose:  Set the current of the selected output                        */
/*=========================================================================*/
int __stdcall tplp_set_current (instrID,opno,current)
int instrID;
int opno;
double current;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (opno, 1, 3, -2) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"I%d %f\012",opno,current);
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return(tplp_err);
    return(tplp_err);  /* success */
}
/*=========================================================================*/
/* Function: Set output status                                             */
/* Purpose:  Set the output status of the selected output On or OFF        */
/*=========================================================================*/
int __stdcall tplp_set_op_stat (instrID,opno,op_stat)
int instrID;
int opno;
int op_stat;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (opno, 1, 3, -2) != 0)
        return(tplp_err);
    if (tplp_invalid_integer_range (op_stat, 0, 1, -3) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"OP%d %d\012",opno,op_stat);
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return(tplp_err);
    return(tplp_err);  /* success */
}
/*=========================================================================*/
/* Function: Set Meter damping                                             */
/* Purpose:  Set the output meter damping of the selected output           */
/*=========================================================================*/
int __stdcall tplp_set_damping (instrID,opno,dmpg)
int instrID;
int opno;
int dmpg;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (opno, 1, 3, -2) != 0)
        return(tplp_err);
    if (tplp_invalid_integer_range (dmpg, 0, 1, -3) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"DAMPING%d %d\012",opno,dmpg);
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return(tplp_err);
    return(tplp_err);  /* success */
}
/*=========================================================================*/
/* Function: Output switch                                                 */
/* Purpose:  Set the master output status On or OFF                        */
/*=========================================================================*/
int __stdcall tplp_op_switch(instrID,status)
int instrID;
int status;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (status, 0, 1, -2) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"OPALL %d\012",status);
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return tplp_err;
    return tplp_err;  /* success */
}
/*=========================================================================*/
/* Function: Set Status                                                    */
/* Purpose:  Set up the IEEE 488.2 status enable registers                 */
/*=========================================================================*/
int __stdcall tplp_set_status (instrID,reg_type,byte_value)
int instrID;
int reg_type;
int byte_value;
{
char str[5];
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (reg_type, 0, 3, -2) != 0)
        return(tplp_err);
    if (tplp_invalid_integer_range (byte_value, 0, 255, -3) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    switch (reg_type) {
    case 0:
        Fmt(str,"*SRE");
        break;
    case 1:
        Fmt(str,"*ESE");
        break;
    case 2:
        Fmt(str,"LSE");
        break;
    case 3:
        Fmt(str,"*PRE");
        break;
    }
    Fmt(buffer,"%s %d\012",str,byte_value);
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return tplp_err;
    return tplp_err;  /* success */
}
/*=========================================================================*/
/* Function:  Read Volts                                                   */
/* Purpose:   Read and return the Voltage at the selected output           */
/*=========================================================================*/
int __stdcall tplp_read_op_volts (instrID,opno,voltage)
int instrID;
int opno;
double *voltage;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (opno, 1, 3, -2) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"V%dO?\012",opno);  /* read voltage */
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return tplp_err;
    if (tplp_read_data(instrID,buffer,50))
        return tplp_err;
    Scan(buffer,"%fV",voltage); /* extract voltage value */
    return tplp_err;  /* success */
}
/*=========================================================================*/
/* Function:  Read current                                                 */
/* Purpose:   Read and return the current at the selected output           */
/*=========================================================================*/
int __stdcall tplp_read_op_current(instrID,opno,current)
int instrID;
int opno;
double *current;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (opno, 1, 3, -2) != 0)
        return(tplp_err);
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(buffer,"I%dO?\012",opno);  /* read current */
    if (tplp_write_data(instrID,buffer,NumFmtdBytes()))
        return tplp_err;
    if (tplp_read_data(instrID,buffer,50))
        return tplp_err;
    Scan(buffer,"%fA",current); /* extract current value */
    return tplp_err;  /* success */
}
/*=========================================================================*/
/* Function:  Read status                                                  */
/* Purpose:   Read and return the IEEE 488.2 status register               */
/*=========================================================================*/
int __stdcall tplp_read_status (instrID,reg_type,byte)
int instrID;
int reg_type;
int *byte;
{
char str[7];
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_invalid_integer_range (reg_type, 0, 6, -1) != 0)
        return tplp_err;
    if (tplp_device_closed (instrID))
        return(tplp_err);

    switch (reg_type) {
    case 0:
        Fmt(str,"*STB?\012");
        break;
    case 1:
        Fmt(str,"*ESR?\012");
        break;
    case 2:
        Fmt(str,"LSR?\012");
        break;
    case 3:
        Fmt(str,"*SRE?\012");
        break;
    case 4:
        Fmt(str,"*ESE?\012");
        break;
    case 5:
        Fmt(str,"LSE?\012");
        break;
    case 6:
        Fmt(str,"*PRE?\012");
        break;
    }
    if (tplp_write_data(instrID,str,NumFmtdBytes()))
         return tplp_err;
    if (tplp_read_data(instrID,buffer,50))
         return tplp_err;
    Scan(buffer,"%d",byte); /* extract value */
    return tplp_err;  /* success */
}
/*=========================================================================*/
/* Function:  Send command                                                 */
/* Purpose:   Send a command to the psu                                    */
/*=========================================================================*/
int __stdcall tplp_send_cmd (instrID,cmd_buf)
int instrID;
char *cmd_buf;
{
char output_buf[2000];
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_device_closed (instrID))
        return(tplp_err);

    Fmt(output_buf,"%s",cmd_buf);
    if (tplp_write_data(instrID,cmd_buf,NumFmtdBytes()))
        return tplp_err;
    return tplp_err;  /* success */
}
/*=========================================================================*/
/* Function:  Get response                                                 */
/* Purpose:   Get and return a response message from the psu               */
/*=========================================================================*/
int __stdcall tplp_get_response (instrID,cmd_buf,count)
int instrID;
char *cmd_buf;
int *count;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;
    if (tplp_device_closed (instrID))
        return(tplp_err);

    if (tplp_read_data(instrID,cmd_buf,2000))
        return tplp_err;
    *count = ibcnt;
    return tplp_err;
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int __stdcall tplp_close (instrID)
int instrID;
{
    if (tplp_invalid_integer_range (instrID, 1, tplp_MAX_INSTR, -1) != 0)
        return tplp_err;

    if (tplp_device_closed (instrID))
        return tplp_err;

    tplp_close_instr (instrID);
    return tplp_err;
}
/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           tplp_MAX_INSTR.                                             */
/*=========================================================================*/
int tplp_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= tplp_MAX_INSTR; i++)
        if (address[i] == addr) {
            instrID = i;
            i = tplp_MAX_INSTR;
         }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= tplp_MAX_INSTR; i++)
            if (address[i] == 0) {
                instrID = i;
                i = tplp_MAX_INSTR;
             }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0) {
        tplp_err = 220;
        return -1;
     }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)    {
        if (instr_cnt <= 0)
            CloseInstrDevs("tt1906");
        bd[instrID] = OpenDev ("", "tt1906");
        if (bd[instrID] <= 0)    {
            tplp_err = 220;
            return -1;
        }
        instr_cnt += 1;
        address[instrID] = addr;
     }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)   {
        tplp_err = 233;
        return -1;
    }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int tplp_close_instr (instrID)
int instrID;
{
    if (bd[instrID] != 0) {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
     }
    else
        tplp_err = 221;

    return tplp_err;
 }
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tplp_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max)) {
    tplp_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int tplp_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)  {
        tplp_err = 232;
        return -1;
    }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tplp_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        tplp_err = 231;
    else
        tplp_err = 0;

    return tplp_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tplp_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        tplp_err = 230;
    else
        tplp_err = 0;

    return tplp_err;
}
/*= THE END ===============================================================*/
